<?php

/**
 * Shopgate GmbH
 *
 * URHEBERRECHTSHINWEIS
 *
 * Dieses Plugin ist urheberrechtlich geschützt. Es darf ausschließlich von Kunden der Shopgate GmbH
 * zum Zwecke der eigenen Kommunikation zwischen dem IT-System des Kunden mit dem IT-System der
 * Shopgate GmbH über www.shopgate.com verwendet werden. Eine darüber hinausgehende Vervielfältigung, Verbreitung,
 * öffentliche Zugänglichmachung, Bearbeitung oder Weitergabe an Dritte ist nur mit unserer vorherigen
 * schriftlichen Zustimmung zulässig. Die Regelungen der §§ 69 d Abs. 2, 3 und 69 e UrhG bleiben hiervon unberührt.
 *
 * COPYRIGHT NOTICE
 *
 * This plugin is the subject of copyright protection. It is only for the use of Shopgate GmbH customers,
 * for the purpose of facilitating communication between the IT system of the customer and the IT system
 * of Shopgate GmbH via www.shopgate.com. Any reproduction, dissemination, public propagation, processing or
 * transfer to third parties is only permitted where we previously consented thereto in writing. The provisions
 * of paragraph 69 d, sub-paragraphs 2, 3 and paragraph 69, sub-paragraph e of the German Copyright Act shall remain unaffected.
 *
 * @author Shopgate GmbH <interfaces@shopgate.com>
 */
class ShopgatePriceModel extends Shopgate_Model_Catalog_Price
{
    
    private $exchangeRate;
    
    /**
     * @var ShopgateConfigGambioGx $config
     */
    private $config;
    
    /**
     * @var int
     */
    private $languageId;
    
    /**
     * @var ShopgateLogger
     */
    private $log;
    
    /**
     * @param ShopgateConfigGambioGx $config
     * @param int                    $languageId
     * @param int                    $exchangeRate
     */
    function __construct(ShopgateConfigGambioGx $config, $languageId, $exchangeRate)
    {
        parent::__construct();
        $this->exchangeRate = $exchangeRate;
        $this->config       = $config;
        $this->languageId   = $languageId;
        $this->log          = ShopgateLogger::getInstance();
    }
    
    
    /**
     * read the max price data to customer groups from the database
     *
     * @return mixed
     */
    public function getCustomerGroupMaxPriceDiscount()
    {
        $this->log("execute SQL customer group ...", ShopgateLogger::LOGTYPE_DEBUG);
        
        $result = array(
            'customerGroupMaxDiscount'           => "",
            'customerGroupMaxDiscountAttributes' => "",
        );
        
        // get customer-group first
        $qry = "SELECT"
            . " status.customers_status_name,"
            . " status.customers_status_discount,"
            . " status.customers_status_discount_attributes"
            . " FROM " . TABLE_CUSTOMERS_STATUS . " AS status"
            . " WHERE status.customers_status_id = " . DEFAULT_CUSTOMERS_STATUS_ID
            . " AND status.language_id = " . $this->languageId
            . ";";
        
        // Check if the customer group exists (ignore if not)
        $queryResult = xtc_db_query($qry);
        if ($queryResult) {
            $customerGroupResult = xtc_db_fetch_array($queryResult);
            if (!empty($customerGroupResult) && isset($customerGroupResult['customers_status_discount'])) {
                $result['customerGroupMaxDiscount'] = $customerGroupResult['customers_status_discount'];
                $result['customerGroupMaxDiscountAttributes'] =
                    $customerGroupResult['customers_status_discount_attributes'] ? true : false;
            }
        }
        
        return $result;
    }
    
    /**
     * calculate the price to an product regarding special prices etc.
     *
     * @param array  $item
     * @param double $tax_rate
     * @param float  $price
     * @param float  $oldPrice
     * @param float  $productDiscount
     * @param bool   $withCustomerGroup
     * @param bool   $withTax
     */
    public function getPriceToItem(
        $item, $tax_rate, &$price, &$oldPrice, &$productDiscount, $withCustomerGroup = true, $withTax = true
    ) {
        $customerGroupMaxPriceDiscount = 0;
        if ($withCustomerGroup) {
            $result                        = $this->getCustomerGroupMaxPriceDiscount();
            $customerGroupMaxPriceDiscount = $result['customerGroupMaxDiscount'];
        }
        
        $price    = $item["products_price"];
        $oldPrice = '';
        
        // Special offers for a Customer group
        $pOffers = $this->getPersonalOffersPrice($item);
        if (!empty($pOffers) && round($pOffers, 2) > 0) {
            $price = $pOffers;
            // Gambio also tells the old price if the offer is lower than the actual price for the customer group
            if ($pOffers < $item["products_price"]) {
                $oldPrice = $item["products_price"];
            }
        }
        
        // General special offer or customer group price reduction
        if (!empty($item["specials_new_products_price"])) {
            if (STOCK_CHECK == 'true' && STOCK_ALLOW_CHECKOUT == 'false') {
                if ($item["specials_quantity"] > 0) {
                    // Nur wenn die quantity > 0 ist dann specialprice setzen,
                    // ansonsten normalen Preis mit normalem Stock
                    $item["products_quantity"] =
                        $item["specials_quantity"] > $item["products_quantity"] ? $item["products_quantity"]
                            : $item["specials_quantity"];
                }
            }
            // setting specialprice
            $oldPrice = $item["products_price"];
            $price    = $item["specials_new_products_price"];
            
            $orderInfos['is_special_price'] = 1;
            
        } elseif (!empty($customerGroupMaxPriceDiscount) && round($customerGroupMaxPriceDiscount, 2) > 0
            && !empty($item['products_discount_allowed'])
            && round($item['products_discount_allowed'], 2) > 0
        ) {
            $productDiscount = round($item['products_discount_allowed'], 2);
            
            // Limit discount to the customer groups maximum discount
            if (round($customerGroupMaxPriceDiscount, 2) < $productDiscount) {
                $productDiscount = round($customerGroupMaxPriceDiscount, 2);
            }
            
            // IMPORTANT NOTICE: GambioGX shoppingsystem takes the higher price value between the offers and the default price,
            // when a discount price is set and then subtracts the discount set for the article
            if ($item["products_price"] > $price) {
                $price = $item["products_price"];
            }
            $oldPrice = $price;
            if ($oldPrice < $item['products_price']) {
                $oldPrice = $item['products_price'];
            }
            
            // Reduce price to the discounted price
            $price = $this->getDiscountPrice($price, $productDiscount);
        }
        $price *= $this->exchangeRate;
        if ($withTax) {
            $price = $price * (1 + ($tax_rate / 100));
        }
        
        if (!empty($oldPrice)) {
            $oldPrice = $oldPrice * $this->exchangeRate;
            if ($withTax) {
                $oldPrice = $oldPrice * (1 + ($tax_rate / 100));
            }
        }
    }
    
    /**
     * Takes a price value and a discount percent value and returns the new discounted price
     *
     * @param float $price
     * @param float $discountPercent
     *
     * @return float
     */
    public function getDiscountPrice($price, $discountPercent)
    {
        $discountedPrice = $price * (1 - $discountPercent / 100);
        
        return $discountedPrice;
    }
    
    /**
     * read the offer price from the database regarding the customer group
     *
     * @param mixed[] $product
     *
     * @return float
     */
    private function getPersonalOffersPrice($product)
    {
        $this->log("execute _getPersonalOffersPrice() ...", ShopgateLogger::LOGTYPE_DEBUG);
        
        $customerStatusId = DEFAULT_CUSTOMERS_STATUS_ID;
        if (empty($customerStatusId)) {
            return false;
        }
        
        $qry = "SELECT * FROM " . TABLE_PERSONAL_OFFERS_BY . "$customerStatusId
        WHERE products_id = '" . $product["products_id"] . "'
        AND quantity = 1";
        
        $qry = xtc_db_query($qry);
        if (!$qry) {
            return false;
        }
        
        $specialOffer = xtc_db_fetch_array($qry);
        
        return floatval($specialOffer["personal_offer"]);
    }
} 
